<?php
session_start();
require 'config.php';

if (!isset($_SESSION['access_token'])) {
    header("Location: login.php");
    exit;
}

$accessToken = $_SESSION['access_token'];
$playlistId = $_GET['playlist_id'] ?? null;

if (!$playlistId) {
    echo "No playlist selected.";
    exit;
}

function getPlaylistTracks($accessToken, $playlistId) {
    $tracks = [];
    $url = "https://api.spotify.com/v1/playlists/$playlistId/tracks?limit=100";

    do {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer $accessToken"
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $response = curl_exec($ch);
        curl_close($ch);

        $data = json_decode($response, true);
        $tracks = array_merge($tracks, $data['items']);
        $url = $data['next'];
    } while ($url);

    return $tracks;
}

$trackItems = getPlaylistTracks($accessToken, $playlistId);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Tracks in Playlist</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/drawflow/dist/drawflow.min.css">
    <script src="https://cdn.jsdelivr.net/npm/drawflow/dist/drawflow.min.js"></script>
</head>
<body class="bg-dark text-white">

<div class="container py-5">
    <div id="drawflow" style="width: 100%; height: 600px;"></div>

    <h2 class="mb-4 text-center">🎶 Tracks in Playlist</h2>
    <div class="text-center mb-4">
        <a href="index.php" class="btn btn-outline-light">← Back to Playlists</a>
    </div>

    <div class="table-responsive">
        <table class="table table-dark table-striped align-middle text-center">
            <thead>
                <tr>
                    <th>Cover</th>
                    <th>Title</th>
                    <th>Artist(s)</th>
                    <th>Album</th>
                    <th>Duration</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($trackItems as $item): 
                    $track = $item['track'];
                    if (!$track || !$track['id']) continue;

                    $spotifyId = $track['id'];
                    $trackName = $track['name'];
                    $artistName = $track['artists'][0]['name'] ?? '';
                    $albumName = $track['album']['name'];
                    $cover = $track['album']['images'][2]['url'] ?? '';
                    $duration = gmdate("i:s", ($track['duration_ms'] / 1000));
                    $fullString = htmlspecialchars($artistName . ' - ' . $trackName);

                    // Insert base track info if not present
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM tracks WHERE spotify_id = ?");
                    $stmt->execute([$spotifyId]);
                    if ($stmt->fetchColumn() == 0) {
                        $insert = $pdo->prepare("
                            INSERT INTO tracks (spotify_id, title, artist, album, duration)
                            VALUES (?, ?, ?, ?, ?)
                        ");
                        $insert->execute([
                            $spotifyId,
                            $trackName,
                            $artistName,
                            $albumName,
                            $duration
                        ]);
                    }

                    // Check if already enriched
                    $stmt = $pdo->prepare("SELECT camelot FROM tracks WHERE spotify_id = ?");
                    $stmt->execute([$spotifyId]);
                    $existing = $stmt->fetch(PDO::FETCH_ASSOC);
                    $isInDb = !empty($existing['camelot']);
                ?>
                    <tr>
                        <td><img src="<?= $cover ?>" width="50"></td>
                        <td><?= htmlspecialchars($trackName) ?></td>
                        <td><?= htmlspecialchars(implode(', ', array_map(fn($a) => $a['name'], $track['artists']))) ?></td>
                        <td><?= htmlspecialchars($albumName) ?></td>
                        <td><?= $duration ?></td>
                        <td>
                            <div class="d-grid gap-2">
                                <button class="btn btn-sm btn-outline-success run-ytmdl-btn" data-track="<?= $fullString ?>">
                                    Download via YTMDL
                                </button>

                                <?php if ($isInDb): ?>
                                    <span class="badge bg-success">✔ Tunebat Cached</span>
                                <?php else: ?>
                                    <button class="btn btn-sm btn-outline-info tunebat-btn"
                                            data-id="<?= $spotifyId ?>"
                                            data-title="<?= htmlspecialchars($trackName) ?>"
                                            data-artist="<?= htmlspecialchars($artistName) ?>">
                                        Fetch Data from Tunebat
                                    </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<script>
document.querySelectorAll('.run-ytmdl-btn').forEach(button => {
    button.addEventListener('click', () => {
        const track = button.getAttribute('data-track');

        fetch('run_ytmdl.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ track })
        })
        .then(res => res.json())
        .then(data => alert(data.message))
        .catch(err => alert('Error triggering download'));
    });
});

document.querySelectorAll('.tunebat-btn').forEach(button => {
    button.addEventListener('click', () => {
        const id = button.getAttribute('data-id');
        const title = button.getAttribute('data-title');
        const artist = button.getAttribute('data-artist');

        button.disabled = true;
        button.innerText = "Fetching from Tunebat...";

        fetch(`trigger-tunebat-fetch.php?id=${encodeURIComponent(id)}&title=${encodeURIComponent(title)}&artist=${encodeURIComponent(artist)}`)
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                button.innerText = "✔ Cached";
                button.classList.remove("btn-outline-info");
                button.classList.add("btn-success");
            } else {
                button.innerText = "❌ Failed";
                button.disabled = false;
            }
        })
        .catch(err => {
            button.innerText = "❌ Error";
            button.disabled = false;
        });
    });
});
</script>

</body>
</html>
